terraform {
  required_providers {
    vcd = {
      source = "vmware/vcd"
    }
  }
  required_version = ">= 0.13"
}

# variables
variable "vcd_user" {}
variable "vcd_org" {}
variable "vcd_url" {}
variable "vcd_allow_unverified_ssl" {
    default = true
}

variable "vdc_password" {
  type        = string
  description = "Enter password for vcd user"
  sensitive = true
}


locals {
  vcd_vdc					= "SiteFactory_VDC04"
  vdc_edge_name				= "SiteFactory_VDC04_Edge01"
  vdcid						= "VDC04"
  vdc_vms_prefix			= "c04"
  project_id				= "p03"
  project_title				= "Sberpro"
  openproject_url			= "https://openproject.appworks.ru/projects/sberpro/"
  network_router_ip			= "10.4.1.65"
  network_bitmask			= "27"
  network_dns1				= "10.255.254.249"
#  network_dns2				= "8.8.8.8"
  network_dns_suffix		= "sitefactory.local"
  network_start_ip			= "10.4.1.66"
  network_end_ip			= "10.4.1.94"
  vm_template_name			= "CentOS8_kernel_4.18.0-193_2021-04-13"
  vm_catalog				= "SiteFactory"
  
  ##################### Resources
  docker_cpu_count			= 2
  docker_ram				= 4096
  docker_hdd				= 6144
  db_cpu_count				= 4
  db_ram					= 3072
  db_hdd					= 10240
  
# Other compute res should be changed below
}

# Configure the VMware vCloud Director Provider
provider "vcd" {
  user                 = var.vcd_user
  password             = var.vdc_password
  org                  = var.vcd_org
  vdc                  = local.vcd_vdc
  url                  = var.vcd_url
  allow_unverified_ssl = var.vcd_allow_unverified_ssl
}

data "vcd_nsxt_edgegateway" "existing" {
  name = local.vdc_edge_name
}

resource "vcd_network_routed_v2" "vapp_routed_network" { ### should be changed
  org         		= var.vcd_org
  vdc         		= local.vcd_vdc
  edge_gateway_id	= data.vcd_nsxt_edgegateway.existing.id

  name          = "PROD_${local.vdcid}_${local.project_id}"
  description   = "Project ${local.project_id} Network (${local.project_title})"
  gateway       = local.network_router_ip
  prefix_length = local.network_bitmask
  dns1          = local.network_dns1
#  dns2          = local.network_dns2
  dns_suffix    = local.network_dns_suffix

  static_ip_pool {
    start_address = local.network_start_ip
    end_address   = local.network_end_ip
  }
}

# Create the vApp
resource "vcd_vapp" "terraform_created_vapp" {
  name			= "PROD_${local.vdcid}_${local.project_id}"
  description   = "Project ${local.project_id} vApp (${local.project_title}). See ${local.openproject_url}"
  power_on		= false
  depends_on	= [vcd_network_routed_v2.vapp_routed_network]
}

resource "vcd_vapp_org_network" "test-vapp-org-network" {
  org				= var.vcd_org
  vdc				= local.vcd_vdc
  vapp_name         = vcd_vapp.terraform_created_vapp.name
  org_network_name  = vcd_network_routed_v2.vapp_routed_network.name
  depends_on		= [vcd_vapp.terraform_created_vapp]
}

# Create the VM in the vApp
resource "vcd_vapp_vm" "dockers" {
  for_each      = toset([ "01", "02", "03"])
  vapp_name 	= vcd_vapp.terraform_created_vapp.name
  catalog_name	= local.vm_catalog
  template_name = local.vm_template_name
  power_on		= false
  
  name			= "${local.vdc_vms_prefix}-${local.project_id}-d-${each.value}"
  computer_name = "${local.vdc_vms_prefix}-${local.project_id}-d-${each.value}"
  memory		= local.docker_ram
  cpus			= local.docker_cpu_count
  cpu_cores		= 1

  metadata = {
    role = "docker"
	project = local.project_id
  }

  network {
    type               = "org"
    name               = vcd_network_routed_v2.vapp_routed_network.name
    ip_allocation_mode = "POOL"
    is_primary         = true
  }
  depends_on			= [vcd_vapp.terraform_created_vapp]
}

resource "vcd_vm_internal_disk" "docker_disk" {
  for_each              = toset([ "01", "02", "03"])
  size_in_mb            = local.docker_hdd
#  storage_profile       = "Tier B"

  vapp_name             = vcd_vapp.terraform_created_vapp.name
  vm_name               = "${local.vdc_vms_prefix}-${local.project_id}-d-${each.value}"
  bus_type              = "paravirtual"
  bus_number            = 0
  unit_number           = 1
  allow_vm_reboot       = true
  depends_on			= [vcd_vapp_vm.dockers]
}

resource "vcd_vapp_vm" "db" {
  vapp_name		= vcd_vapp.terraform_created_vapp.name
  catalog_name	= local.vm_catalog
  template_name = local.vm_template_name
  power_on		= false

  name					= "${local.vdc_vms_prefix}-${local.project_id}-db"
  computer_name 		= "${local.vdc_vms_prefix}-${local.project_id}-db"
  storage_profile       = "Tier 1"
  memory				= local.db_ram
  cpus					= local.db_cpu_count
  cpu_cores				= 1

  metadata = {
    role = "db"
	project = local.project_id
  }

  network {
    type               = "org"
    name               = vcd_network_routed_v2.vapp_routed_network.name
    ip_allocation_mode = "POOL"
    is_primary         = true
  }
  depends_on			= [vcd_vapp.terraform_created_vapp]
}

resource "vcd_vm_internal_disk" "db_disk" {
  size_in_mb            = local.db_hdd
#  storage_profile       = "Tier 1"

  vapp_name             = vcd_vapp.terraform_created_vapp.name
  vm_name               = "${local.vdc_vms_prefix}-${local.project_id}-db"
  bus_type              = "paravirtual"
  bus_number            = 0
  unit_number           = 1
  allow_vm_reboot       = true
  depends_on			= [vcd_vapp_vm.db]
}